using System;
using System.Text.RegularExpressions;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This is a datatype for various timeouts used throughout VistALink library. 
	/// It implements simplistic check (timeout must be a positive integer less than MaxInt / 1000 milliseconds)
	/// and explicit conversion to seconds/milliseconds (to avoid confusion). 
	/// </summary>
	public sealed class Timeout
	{
		private const int MsMultiplier = 1000;
		private int _valueInMilliseconds;
		private static readonly Regex _isNumber = new Regex( @"\A\d+\Z" );

		/// <summary>
		/// The only instance constructor performing check for the 
		/// value to be used as timeout (it must be positive integer). 
		/// </summary>
		/// <param name="intervalInMilliseconds">Timeout interval in milliseconds.</param>
		public Timeout( int intervalInMilliseconds )
		{
			string errorMessage;

			if( !IsValidTimeoutValueInMilliseconds( intervalInMilliseconds, out errorMessage ) )
				throw( new ArgumentOutOfRangeException( "intervalInMilliseconds", SR.Exceptions.TimeoutSuppliedValueIsInvalid( errorMessage ) ) );

			_valueInMilliseconds = intervalInMilliseconds;
		}

		/// <summary>
		/// Static timeout constructor parsing input string, verifying that it satisfies 
		/// to timeout requirements and returning timeout instance.
		/// </summary>
		/// <param name="intervalInSeconds">Timeout interval in seconds as string.</param>
		/// <returns>New instance of timeout created from supplied string.</returns>
		public static Timeout ParseStringInSeconds( string intervalInSeconds )
		{
			if( intervalInSeconds == null ) 
				throw( new ArgumentNullException( "intervalInSeconds", SR.Exceptions.TimeoutInputStringIsNull() ) );

			string errorMessage;

			if( !IsValidTimeoutInSecondsString( intervalInSeconds, out errorMessage ) )
				throw( new StringParseException( SR.Exceptions.TimeoutSuppliedValueIsInvalid( errorMessage ) ) );

			checked
			{
				return new Timeout( Convert.ToInt32( intervalInSeconds ) * MsMultiplier );
			}
		}

		/// <summary>
		/// Checks if supplied numeric value can be converted to timeout value in milliseconds. 
		/// Value must be greater than zero and less than Int32.MaxValue.
		/// Provides descriptive message via out parameter if there are problems encountered with the value. 
		/// </summary>
		/// <param name="testTimeoutValueInMilliseconds">Numeric value to test.</param>
		/// <param name="errorMessage">Error message.</param>
		/// <returns>
		///		True and sets out error message to null if test value conforms to timeout value requirements. 
		///		Returns false and sets out descriptive error message otherwise.
		///	</returns>
		public static bool IsValidTimeoutValueInMilliseconds( int testTimeoutValueInMilliseconds, out string errorMessage )
		{
			if( testTimeoutValueInMilliseconds <= 0 )
			{
				errorMessage = SR.Exceptions.TimeoutInputValueIsNotPositiveInteger();
				return false;
			}

			errorMessage = null;
			return true;
		}

		/// <summary>
		/// Checks if supplied string value can be converted to timeout value. 
		/// Provides descriptive message via out parameter if there are problems encountered with the value. 
		/// </summary>
		/// <param name="testTimeoutInSecondsString">String value to test.</param>
		/// <param name="errorMessage">Error message.</param>
		/// <returns>
		///		True and sets out error message to null if test value conforms to timeout value requirements. 
		///		Returns false and sets out descriptive error message otherwise.
		///	</returns>
		public static bool IsValidTimeoutInSecondsString( string testTimeoutInSecondsString, out string errorMessage )
		{	
			if( !CanBeConvertedToNaturalNumber( testTimeoutInSecondsString, out errorMessage ) )
				return false;

			int _testValue;

			try
			{
				_testValue = Convert.ToInt32( testTimeoutInSecondsString );
			}
			catch( OverflowException )
			{
				errorMessage = SR.Exceptions.TimeoutValueInInputStringIsTooLarge( Int32.MaxValue );
				return false;
			}

			if( !IsValidTimeoutValueInMilliseconds( _testValue, out errorMessage ) )
				return false;

			// additional limitation is put to timeout in seconds (because it will be multiplied by MsMultiplier)
			if( Convert.ToDouble( _testValue ) * Convert.ToDouble( MsMultiplier ) >= Convert.ToDouble( Int32.MaxValue ) )
			{
				errorMessage = SR.Exceptions.TimeoutInputStringSecondsValueIsTooLarge( Convert.ToInt32( Convert.ToDouble( Int32.MaxValue ) / Convert.ToDouble( MsMultiplier ) ) );
				return false;
			}

			errorMessage = null;
			return true;
		}

		/// <summary>
		/// This method checks if string can be converted to natural number. 
		/// </summary>
		/// <param name="testValueString">String value to test.</param>
		/// <param name="errorMessage">Output error message.</param>
		///		True and sets out error message to null if test value can be converted to natural number.
		///		Returns false and sets out descriptive error message otherwise.
		private static bool CanBeConvertedToNaturalNumber( string testValueString, out string errorMessage )
		{
			if( testValueString == null )
			{
				errorMessage = SR.Exceptions.TimeoutInputStringIsNull();
				return false;
			}

			if( !_isNumber.IsMatch( testValueString ) )
			{
				errorMessage = SR.Exceptions.TimeoutInputStringIsNotNumeric();
				return false;
			}

			errorMessage = null;
			return true;
		}
		
		/// <summary>
		/// This method returns timeout value converted into seconds as int. 
		/// </summary>
		/// <returns>Timeout value converted into seconds as int.</returns>
		public int ToInt32Seconds()
		{
			checked
			{
				return Convert.ToInt32( Convert.ToDouble( _valueInMilliseconds ) / Convert.ToDouble( MsMultiplier ) );
			}
		}

		/// <summary>
		/// This method returns timeout value converted into seconds as string. 
		/// </summary>
		/// <returns>Timeout value converted into seconds as string.</returns>
		public string ToStringSeconds()
		{
			return this.ToInt32Seconds().ToString();
		}

		/// <summary>
		/// This method returns timeout value converted into milliseconds as int. 
		/// </summary>
		/// <returns>Timeout value converted into milliseconds as int.</returns>
		public int ToInt32Milliseconds()
		{
			return _valueInMilliseconds;
		}	
	}
}
